import { Injectable } from '@angular/core';


export interface PersonTaskData {
  task: string;
  project: string;
  user: string;
  start: Date;
  completed: Date;
  status: string;
  percent: number;
  parent: string;
}

@Injectable({
  providedIn: 'root'
})

export class DataService {
  // PersonTaskDataのグローバル管理用
  private _personTaskDataMap = new Map<string, PersonTaskData[]>();

  constructor() { }
    // gets the holiday for a given date
  private readonly _holidays = {
        '1/1': '元旦',
        '1/2/1': '成人の日',
        '2/11': '建国記念の日',
        '2/23': '天皇誕生日',
        '3/20': '春分の日',
        '4/29': '昭和の日',
        '5/3': '憲法記念日',
        '5/4': 'みどりの日',
        '5/5': 'こどもの日',
        '7/3/1': '海の日',
        '8/11': '山の日',
        '9/23': '秋分の日',
        '9/3/1': '敬老の日',
        '10/2/1': '体育の日',
        '11/3': '文化の日',
        '11/23': '勤労感謝の日'
    };
    //

    // 部門別のタスクと関連プロジェクト
    readonly departmentTasks = {
      '営業部': {
        tasks: [
          '新規顧客開拓営業活動',
          '既存顧客フォローアップ',
          '営業資料作成',
          '営業会議準備',
          '競合他社分析',
          '顧客満足度調査',
          '営業戦略立案',
          '商談資料準備',
          '営業目標管理',
          'クライアント訪問'
        ],
        projects: ['営業強化プロジェクト', '顧客開拓', '売上向上施策', '営業効率化']
      },
      '開発部': {
        tasks: [
          'システム設計書作成',
          'プログラミング実装',
          'バグ修正対応',
          'コードレビュー',
          'テスト仕様書作成',
          '単体テスト実行',
          'システム統合テスト',
          '技術調査・検証',
          'API設計・開発',
          'データベース設計'
        ],
        projects: ['新システム開発', 'レガシー改修', 'API統合', 'パフォーマンス改善']
      },
      'IT部': {
        tasks: [
          'サーバー保守管理',
          'ネットワーク監視',
          'セキュリティ対策',
          'システム障害対応',
          'バックアップ作業',
          'ソフトウェア更新',
          'ユーザーサポート',
          'インフラ構築',
          'クラウド移行作業',
          '監視システム構築'
        ],
        projects: ['ITインフラ強化', 'クラウド移行', 'セキュリティ向上', 'システム運用']
      },
      '経理部': {
        tasks: [
          '月次決算処理',
          '請求書発行',
          '経費精算処理',
          '予算管理',
          '財務分析レポート',
          '税務申告準備',
          '支払い処理',
          '売掛金管理',
          '固定資産管理',
          '監査対応'
        ],
        projects: ['決算業務改善', '経理システム導入', '予算管理強化', '財務分析']
      },
      '人事部': {
        tasks: [
          '採用活動',
          '面接実施',
          '人事評価',
          '給与計算',
          '研修企画・実施',
          '労務管理',
          '就業規則見直し',
          '社内イベント企画',
          '人材配置検討',
          '退職手続き'
        ],
        projects: ['人材採用強化', '人事制度改革', '教育研修充実', '働き方改革']
      },
      'マーケティング部': {
        tasks: [
          'マーケット調査',
          '広告キャンペーン企画',
          'SNS運用',
          'Webサイト更新',
          'イベント企画・運営',
          'プレスリリース作成',
          'ブランディング戦略',
          '顧客分析',
          'コンテンツ制作',
          'SEO対策'
        ],
        projects: ['ブランド認知向上', 'デジタルマーケティング', '顧客獲得施策', 'PR強化']
      },
      '総務部': {
        tasks: [
          '契約書管理',
          '法務相談対応',
          'コンプライアンス管理',
          '社内規程整備',
          'オフィス管理',
          '備品調達',
          '保険手続き',
          '株主総会準備',
          'リスク管理',
          '文書管理'
        ],
        projects: ['コンプライアンス強化', 'オフィス環境改善', '業務効率化', '法務体制整備']
      },
      '品質管理部': {
        tasks: [
          '品質検査',
          '品質基準策定',
          '改善提案作成',
          '監査実施',
          '品質データ分析',
          '不具合原因調査',
          '品質マニュアル更新',
          '供給業者評価',
          '品質会議準備',
          '是正措置対応'
        ],
        projects: ['品質向上施策', 'ISO認証取得', '品質管理システム', '改善活動推進']
      },
      'デザイン部': {
        tasks: [
          'UI/UXデザイン',
          'ロゴデザイン',
          'パンフレット制作',
          'Webデザイン',
          'プロトタイプ作成',
          'デザインガイドライン',
          '画像素材作成',
          'ブランドデザイン',
          'ユーザビリティテスト',
          'デザインレビュー'
        ],
        projects: ['ブランドリニューアル', 'Webサイト刷新', 'UI改善', 'デザインシステム']
      },
      '製造部': {
        tasks: [
          '生産計画立案',
          '製造工程管理',
          '品質チェック',
          '設備保守',
          '在庫管理',
          '安全管理',
          '効率改善',
          '原材料調達',
          '納期管理',
          '作業指示書作成'
        ],
        projects: ['生産効率向上', '設備更新', '安全対策強化', '製造プロセス改善']
      }
    };
  getPersonTaskData(personName: string, department: string): PersonTaskData[] {
    // 既にキャッシュされているデータがあれば返す
    const cachedData = this._personTaskDataMap.get(personName);
    if (cachedData) {
      return cachedData;
    }

    const deptData = this.departmentTasks[department] || this.departmentTasks['営業部'];
    const data: PersonTaskData[] = [];
    const taskCount = Math.floor(Math.random() * 6) + 15; // 15-20件のタスク
    const pairs = this.getRandomPairs(deptData.tasks,  deptData.projects, taskCount);

    for (const pair of pairs) {
      const startDay = Math.floor(Math.random() * 30) + 1;
      const endDay = Math.min(startDay + Math.floor(Math.random() * 7) + 1, 31);

      const createDate = (day: number) => {
        const now = new Date();
        return new Date(now.getFullYear(), now.getMonth(), day); // 現在の年月のDate型
      };
      const isCompleted = Math.random() > 0.7; // 30%の確率で完了
      const isInProgress = !isCompleted && Math.random() > 0.5; // 完了でない場合の50%で進行中
      
      let status = 'not-started';
      let percent = 0;
      
      if (isCompleted) {
        status = 'completed';
        percent = 100;
      } else if (isInProgress) {
        status = 'in-progress';
        percent = Math.floor(Math.random() * 80) + 10; // 10-90%のランダム
      }
      
      data.push({
        task: pair.task,
        project: pair.project,
        user: personName,
        start: createDate(startDay),
        completed: createDate(endDay),
        status: status,
        percent: percent,
        parent: ''
      });
    }
    
    // データをキャッシュ
    this._personTaskDataMap.set(personName, data);
    return data;
  }

  getHoliday(date: Date) {
    const day = date.getDate();
    const month = date.getMonth() + 1;
    let holiday = this._holidays[month + '/' + day];
    if (!holiday) {
      const weekDay = date.getDay();
      const weekNum = Math.floor((day - 1) / 7) + 1;
      holiday = this._holidays[month + '/' + weekNum + '/' + weekDay];
    }
    return holiday;
  }

  getRandomPairs(tasks, projects, count) {
    // 全組み合わせを作成
    const allPairs = [];
    for (const task of tasks) {
        for (const project of projects) {
            allPairs.push({ task, project });
        }
    }

    // Durstenfeldでシャッフル
    for (let i = allPairs.length - 1; i > 0; i--) {
        const j = Math.floor(Math.random() * (i + 1));
        [allPairs[i], allPairs[j]] = [allPairs[j], allPairs[i]];
    }

    return allPairs.slice(0, Math.min(count, allPairs.length));
  }

  getStatusOptions() {
    return ['not-started', 'in-progress', 'completed']
  }

  // PersonTaskDataの特定タスクを更新する
  updatePersonTaskData(personName: string, taskIndex: number, updatedData: Partial<PersonTaskData>): boolean {
    const personTasks = this._personTaskDataMap.get(personName);
    if (personTasks && taskIndex >= 0 && taskIndex < personTasks.length) {
      // 既存のタスクデータを更新
      Object.assign(personTasks[taskIndex], updatedData);
      return true;
    }
    return false;
  }

  // PersonTaskDataの特定タスクのステータスを更新する
  updateTaskStatus(personName: string, task: string, project: string, newStatus: string): boolean {
    const personTasks = this._personTaskDataMap.get(personName);
    if (personTasks) {
      const taskToUpdate = personTasks.find(t => t.task === task && t.project === project);
      if (taskToUpdate) {
        taskToUpdate.status = newStatus;
        // ステータスに応じてパーセンテージも更新
        taskToUpdate.percent = newStatus === 'completed' ? 100 : 
                              newStatus === 'in-progress' ? 60 :
                              newStatus === 'not-started' ? 0 : taskToUpdate.percent;
        return true;
      }
    }
    return false;
  }

  // PersonTaskDataの特定タスクの任意フィールドを更新する（汎用版）
  updateTask(personName: string, task: string, project: string, updates: Partial<PersonTaskData>): boolean {
    const personTasks = this._personTaskDataMap.get(personName);
    if (personTasks) {
      const taskToUpdate = personTasks.find(t => t.task === task && t.project === project);
      if (taskToUpdate) {
        // 提供された更新データをマージ
        Object.assign(taskToUpdate, updates);
        
        // ステータスが変更された場合、パーセンテージも自動調整（ただし、明示的にparentが設定されていない場合のみ）
        if (updates.status && !updates.hasOwnProperty('percent')) {
          taskToUpdate.percent = updates.status === 'completed' ? 100 : 
                                updates.status === 'in-progress' ? 60 : 
                                updates.status === 'not-started' ? 0 : taskToUpdate.percent;
        }
        return true;
      }
    }
    return false;
  }

  // PersonTaskDataに新しいタスクを追加する
  addPersonTaskData(personName: string, newTask: PersonTaskData): boolean {
    const personTasks = this._personTaskDataMap.get(personName);
    if (personTasks) {
      personTasks.push(newTask);
      return true;
    }
    return false;
  }
}


