﻿import { Component, Input, OnInit, OnChanges, SimpleChanges, ViewChild, EventEmitter, Output } from '@angular/core';
import { CommonModule } from '@angular/common';
import { WjChartModule } from '@mescius/wijmo.angular2.chart';
import { DataService, PersonTaskData } from '../data.service';

import { FlexChart, FlexChartCore, IRenderEngine, HitTestInfo, Palettes, RenderEventArgs } from '@mescius/wijmo.chart';
import { clamp, format, isArray, isNumber, Rect } from '@mescius/wijmo';
import { Person } from '../persons-list/persons-list.component';


@Component({
  selector: 'gantt-chart',
  templateUrl: './gantt-chart.component.html',
  styleUrls: ['./gantt-chart.component.css'],
  standalone: true,
  imports: [CommonModule, WjChartModule]
})

export class GanttChartComponent implements OnInit, OnChanges {

    @Input('selectedPerson') selectedPerson: Person | null = null;
    @Input('personTaskData') personTaskData: PersonTaskData[] = null;
    @ViewChild('chart', { static: false }) chart!: FlexChart;
    data;
    palette: string[];

    constructor(private dataService: DataService) {
        this.palette = this._getRandomPalette();
    }

    ngOnInit(): void {
    }

    ngOnChanges(changes: SimpleChanges): void {
        if (changes.personTaskData && this.personTaskData) {
            this.data = this.personTaskData.map(item => ({
                ...item,
                name: `${item.task} - ${item.project}`
            }));
            
            // チャートが初期化されている場合、データを強制的に更新
            if (this.chart) {
                setTimeout(() => {
                    this.chart.invalidate();
                }, 0);
            }
        }
    }

    getTooltipContent(ht: HitTestInfo) {
        const task: PersonTaskData = ht.item;
        let str = format('<b>{task}</b>:<br/>{start:d} - {completed:d}', {
            task: ht.x,
            start: task.start,
            completed: task.completed
        });

        if (task && task.percent != null) {
            str += format('<br/><i>{percent}％完了</i>', task);
        }

        return str;
    }
    // タスクの進捗率を描画する関数
    // defaultFormatter() で通常のタスクバーを描画。
    // task.percent を使って進捗率を計算し、バーの中に進捗を示す矩形を描画。
    ganttItemFormatter = (engine: IRenderEngine, ht: HitTestInfo, defaultFormatter: Function) => {
        // 通常のバーを描画
        defaultFormatter();
        //
        // 進捗率を描画
        const task: PersonTaskData = ht.item;
        //
        if (isNumber(task.percent) && task.percent > 0) {
            const pct = clamp(task.percent, 0, 100) / 100;
            const  rc = this._getTaskRect(ht.series.chart, task).inflate(-8, -8);

            engine.fill = pct === 1 ? '#4caf50' : '#ff9800'; // 完了なら緑、途中なら金色;
            const taskRect = this._getTaskRect(ht.series.chart, task);
            const progressRect = this._getProgressRect(taskRect, pct);
            engine.drawRect(progressRect.left, progressRect.top, progressRect.width, progressRect.height);
        }

    }

    private _getProgressRect(taskRect: Rect, progressPct: number) {
        const shrinkRatio = 0.1; // 上下左右10%余白
        const left = taskRect.left + taskRect.width * shrinkRatio;
        const top = taskRect.top + taskRect.height * shrinkRatio;
        const width = taskRect.width * (1 - shrinkRatio * 2) * progressPct;
        const height = taskRect.height * (1 - shrinkRatio * 2);

        return new Rect(left, top, width, height);
    }
    //
    // タスクの親子関係を表示する
    ganttChartRendered = (chart: FlexChart, e: RenderEventArgs) => {
        const tasks: PersonTaskData[] = chart.collectionView.items;
        tasks.forEach(task => { // for each task
            const parents = this._getTaskParents(task, tasks); // get the parent tasks
            parents.forEach(parent => { // for each parent
                this._drawConnectingLine(e.engine, chart, task, parent); // draw connector
            });
        });
    }


    private _getRandomPalette() {
        const palettes = Object.keys(Palettes).filter(prop => isArray(Palettes[prop]));
        const rand = Math.floor(Math.random() * palettes.length);
        return Palettes[palettes[rand]];
    }

    private _drawConnectingLine(engine: IRenderEngine, chart: FlexChart, task: PersonTaskData, parent: PersonTaskData) {
        const rc1 = this._getTaskRect(chart, parent);   // parent rect
        const rc2 = this._getTaskRect(chart, task);     // task rect
        const x1 = rc1.left + rc1.width / 2;      // parent x center
        const x2 = rc2.left;                      // task left
        const y1 = rc1.bottom;                    // parent bottom
        const y2 = rc2.top + rc2.height / 2;      // task y center
        //
        // draw connecting line
        let xs = [x1, x1, x2];
        let ys = [y1, y2, y2];
        //
        engine.drawLines(xs, ys, 'connector', {
            stroke: 'black'
        });
        //
        // draw arrow at the end
        const sz = 5;
        //
        xs = [x2 - 2 * sz, x2, x2 - 2 * sz];
        ys = [y2 - sz, y2, y2 + sz];
        //
        engine.drawPolygon(xs, ys, 'arrow', {
            fill: 'black'
        });
    }

    private _getTaskRect(chart: FlexChartCore, task: PersonTaskData) {
        const x1 = chart.axisX.convert(task.start.valueOf());
        const x2 = chart.axisX.convert(task.completed.valueOf());
        const index = chart.collectionView.items.indexOf(task);

        const totalTasks = chart.collectionView.items.length;

        // タスク間隔は1なので、バーの高さは (1 - marginRatio)
        const marginRatio = 0.2; // 上下に20%余白
        const barHeightRatio = 1 - marginRatio; // バーの高さ = 80%のスペース
        const factor = barHeightRatio / 2; // 半分ずつ上下に広げる

        const y1 = chart.axisY.convert(index - factor);
        const y2 = chart.axisY.convert(index + factor);

        const top = Math.min(y1, y2);
        const height = Math.abs(y2 - y1);

        return new Rect(x1, top, x2 - x1 + 1, height);
    }

    private _getTaskParents(task: PersonTaskData, tasks: PersonTaskData[]) {
        const parents: PersonTaskData[] = [];

        if (task.parent) {
            task.parent.split(',').forEach(name => {
                for (let i = 0; i < tasks.length; i++) {
                    if (tasks[i].task === name) {
                        parents.push(tasks[i]);
                        break;
                    }
                }
                // tasks.forEach(t => {
                //     if (t.name === name) {
                //         parents.push(t);
                //         break;
                //     }
                // });
            });
        }

        return parents;
    }

    setChartWidth(width: number) {
        if (this.chart) {
            this.chart.hostElement.style.width = `${width}px`;
        }
    }

}
