import { Component, OnInit, Input, OnChanges, SimpleChanges } from '@angular/core';
import { CommonModule } from '@angular/common';
import { WjChartModule } from '@mescius/wijmo.angular2.chart';
import { WjCoreModule } from '@mescius/wijmo.angular2.core';
import { Person } from '../persons-list/persons-list.component';
import { DataService } from '../data.service';
import { HitTestInfo } from '@mescius/wijmo.chart';

@Component({
  selector: 'app-monthly-achievement-chart',
  templateUrl: './monthly-achievement-chart.component.html',
  styleUrls: ['./monthly-achievement-chart.component.css'],
  standalone: true,
  imports: [CommonModule, WjChartModule, WjCoreModule]
})
export class MonthlyAchievementChartComponent implements OnInit, OnChanges {
  @Input('selectedPerson') selectedPerson: Person | null = null;
  
  monthlyAchievementData: any[] = [];
  chartTooltip = '選択された人物の月次作業達成率を表示します。';

  // 系列のスタイル
  lastYearSeriesStyle = {
    strokeWidth: 3,
    stroke: '#0066cc',
    symbolStyle: {
      fill: '#0066cc',
      stroke: '#004499',
      strokeWidth: 2
    }
  };

  thisYearSeriesStyle = {
    strokeWidth: 3,
    stroke: '#ff6600',
    symbolStyle: {
      fill: '#ff6600',
      stroke: '#cc4400',
      strokeWidth: 2
    }
  };

  constructor(private dataService: DataService) { 
    // デフォルトデータ（誰も選択されていない時）
    this.generateDefaultMonthlyData();
  }

  ngOnInit(): void {}

  ngOnChanges(changes: SimpleChanges): void {
    if (changes['selectedPerson'] && this.selectedPerson) {
      this.generateMonthlyAchievementData();
      this.chartTooltip = `${this.selectedPerson.name}の月次作業達成率です。`;
    }
  }

  private generateDefaultMonthlyData(): void {
    this.monthlyAchievementData = [];
    
    // 去年のデータ（2024年1-12月）
    const lastYearData = [75, 82, 78, 85, 88, 92, 87, 90, 85, 89, 94, 91];
    for (let i = 1; i <= 12; i++) {
      this.monthlyAchievementData.push({
        month: `${i}月`,
        lastYearRate: lastYearData[i - 1],
        thisYearRate: null,
        monthNum: i,
        year: 2024
      });
    }
    
    // 今年のデータ（2025年1-9月）
    const thisYearData = [80, 86, 83, 89, 92, 95, 91, 94, 88];
    for (let i = 1; i <= 9; i++) {
      this.monthlyAchievementData[i - 1].thisYearRate = thisYearData[i - 1];
    }
  }

  private generateMonthlyAchievementData(): void {
    if (!this.selectedPerson) return;

    // 人物ごとに異なる達成率パターンを生成
    const lastYearBasePattern = this.getPersonAchievementPattern(this.selectedPerson.department, 2024);
    const thisYearBasePattern = this.getPersonAchievementPattern(this.selectedPerson.department, 2025);
    
    this.monthlyAchievementData = [];
    
    // 12ヶ月分のデータを生成
    for (let i = 1; i <= 12; i++) {
      // 去年のデータ
      const lastYearBaseRate = lastYearBasePattern[i - 1];
      const lastYearPersonalVariation = (Math.random() - 0.5) * 10;
      let lastYearRate = Math.round(lastYearBaseRate + lastYearPersonalVariation);
      lastYearRate = Math.max(0, Math.min(100, lastYearRate));
      
      // 今年のデータ（1-9月のみ）
      let thisYearRate = null;
      if (i <= 9) {
        const thisYearBaseRate = thisYearBasePattern[i - 1];
        const thisYearPersonalVariation = (Math.random() - 0.5) * 10;
        thisYearRate = Math.round(thisYearBaseRate + thisYearPersonalVariation);
        thisYearRate = Math.max(0, Math.min(100, thisYearRate));
      }
      
      this.monthlyAchievementData.push({
        month: `${i}月`,
        lastYearRate: lastYearRate,
        thisYearRate: thisYearRate,
        monthNum: i
      });
    }
  }

  private getPersonAchievementPattern(department: string, year: number = 2024): number[] {
    // 部門ごとの基本達成率パターン
    const lastYearPatterns: { [key: string]: number[] } = {
      'IT部': [70, 75, 80, 85, 88, 90, 87, 92, 89, 91, 95, 93],
      '営業部': [65, 72, 78, 82, 85, 88, 84, 87, 85, 89, 92, 90],
      '人事部': [72, 78, 75, 80, 83, 87, 85, 88, 86, 90, 93, 91],
      '経理部': [68, 74, 77, 81, 84, 86, 83, 89, 87, 88, 91, 89],
      '総務部': [70, 76, 79, 83, 86, 89, 85, 90, 88, 92, 94, 92]
    };
    
    // 今年は去年より少し向上したパターン
    const thisYearPatterns: { [key: string]: number[] } = {
      'IT部': [75, 80, 85, 88, 91, 93, 90, 95, 92],
      '営業部': [70, 77, 83, 87, 90, 92, 88, 91, 89],
      '人事部': [77, 83, 80, 85, 88, 92, 90, 93, 91],
      '経理部': [73, 79, 82, 86, 89, 91, 88, 94, 92],
      '総務部': [75, 81, 84, 88, 91, 94, 90, 95, 93]
    };
    
    if (year === 2025) {
      return thisYearPatterns[department] || thisYearPatterns['IT部'];
    } else {
      return lastYearPatterns[department] || lastYearPatterns['IT部'];
    }
  }

  tooltipContent(hti: HitTestInfo) {
    let item = hti.item;
    let year = item.lastYearRate === hti.y ? 2024 : 2025;
    return `<b>${year} 年${hti.x} 月</b></br><b>達成率: ${hti.y} ％</b>`;
  } 
}
