﻿import { Component, Input, Output, EventEmitter, OnInit, OnChanges, SimpleChanges, ViewChild, NO_ERRORS_SCHEMA } from '@angular/core';
import { CommonModule } from '@angular/common';
import { WjGridModule } from '@mescius/wijmo.angular2.grid';
import { WjCoreModule } from '@mescius/wijmo.angular2.core';
import { WjInputModule } from '@mescius/wijmo.angular2.input';
import { DataService, PersonTaskData } from '../data.service';
import { Person } from '../persons-list/persons-list.component';

import * as wjCore from '@mescius/wijmo';
import { FlexGrid ,CellEditEndingEventArgs} from '@mescius/wijmo.grid';

export interface UserInfo {
  task: string;
  project: string;
  person: Person;
  start: Date;
  completed: Date;
  status: string;
}

@Component({
  selector: 'app-user-info',
  templateUrl: './user-info.component.html',
  styleUrls: ['./user-info.component.css'],
  standalone: true,
  imports: [CommonModule, WjInputModule, WjGridModule, WjCoreModule],
  schemas: [NO_ERRORS_SCHEMA]
})
export class UserInfoComponent implements OnInit, OnChanges {
  @Input('selectedCountryName') selectedCountryName: string;
  @Input('selectedPerson') selectedPerson: Person | null = null;
  @Input('personTaskData') personTaskData: PersonTaskData[] = null;
  @Output() taskDataChanged = new EventEmitter<Person>();
  @ViewChild('flexGrid', { static: false }) flexGrid!: FlexGrid;

  selectedCountryData: wjCore.CollectionView;
  usersTooltip = 'プロジェクトのタスク一覧とその進捗状況です。';
  status = this.dataService.getStatusOptions();
  availableTasks: string[] = [];
  availableProjects: string[] = [];
  minEndDate: Date | null = null;

  constructor(private dataService: DataService) {
  }

  ngOnInit(): void {
    this.selectedCountryData = new wjCore.CollectionView(this.personTaskData, {
      pageSize: 5
    });

    this.usersTooltip = '仙台 太郎（営業部）のタスク一覧です。';
  }

  ngOnChanges(changes: SimpleChanges): void {
    if (changes['selectedPerson'] && this.selectedPerson) {
      // 選択された人物のタスクデータを取得
      this.selectedCountryData = new wjCore.CollectionView(this.personTaskData, {
        pageSize: 10
      });

      // ツールチップも更新
      this.usersTooltip = `${this.selectedPerson.name}（${this.selectedPerson.department}）のタスク一覧です。`;
      
      // 選択された人物の部門に基づいてタスクとプロジェクトリストを更新
      this.updateAvailableTasksAndProjects();
    }
  }
  
  private updateAvailableTasksAndProjects(): void {
    if (!this.selectedPerson) return;
    
    const deptData = this.dataService.departmentTasks[this.selectedPerson.department];
    if (deptData) {
      this.availableTasks = [...deptData.tasks];
      this.availableProjects = [...deptData.projects];
    } else {
      // デフォルトとして営業部のデータを使用
      const defaultDept = this.dataService.departmentTasks['営業部'];
      this.availableTasks = [...defaultDept.tasks];
      this.availableProjects = [...defaultDept.projects];
    }
  }
  //グリッドのセルが変更されたときのイベント
  flexInitialized(flexgrid: FlexGrid) {
    flexgrid.cellEditEnding.addHandler((s: FlexGrid, e: CellEditEndingEventArgs) => {
      let col = s.columns[e.col];
      let oldVal = flexgrid.getCellData(e.row, e.col, false);
      let newVal = flexgrid.activeEditor.value;
      
      // 値が変更されている場合
      if (oldVal !== newVal) {
        let item = s.rows[e.row].dataItem as PersonTaskData;
        let updates: Partial<PersonTaskData> = {};
        
        // フィールドに応じて更新データを設定（適切な型変換を行う）
        if (col.binding === 'status') {
          item.status = newVal;
          updates.status = newVal;
          // ステータス変更時にパーセンテージも自動更新
          const autoPercent = newVal === 'completed' ? 100 : 
                             newVal === 'in-progress' ? 60 : 
                             newVal === 'not-started' ? 0 : item.percent;
          item.percent = autoPercent;
          updates.percent = autoPercent;
        } else if (col.binding === 'start') {
          const startDate = new Date(newVal);
          item.start = startDate;
          updates.start = startDate;
        } else if (col.binding === 'completed') {
          const completedDate = new Date(newVal);
          item.completed = completedDate;
          updates.completed = completedDate;
        } else if (col.binding === 'percent') {
          const percentValue = Number(newVal);
          item.percent = percentValue;
          updates.percent = percentValue;
          
          // percent値に応じてstatusも自動更新
          let autoStatus = item.status; // デフォルトは現在のステータス
          if (percentValue === 100) {
            autoStatus = 'completed';
          } else if (percentValue === 0) {
            autoStatus = 'not-started';
          } else if (percentValue > 0 && percentValue < 100) {
            autoStatus = 'in-progress';
          }
          
          if (autoStatus !== item.status) {
            item.status = autoStatus;
            updates.status = autoStatus;
          }
        }
        
        // DataServiceのグローバルデータも更新
        if (this.selectedPerson && Object.keys(updates).length > 0) {
          this.dataService.updateTask(
            this.selectedPerson.name,
            item.task,
            item.project,
            updates
          );
          
          // 親コンポーネントに変更を通知
          this.taskDataChanged.emit(this.selectedPerson);
        }
      }
    });
  }

  getFlexGridWidth(): number {
    return this.flexGrid ? this.flexGrid.hostElement.offsetWidth : 0;
  }

  onStartDateChanged(startDate: Date): void {    
    if (startDate && startDate instanceof Date) {
      // 開始日の翌日を最小終了日として設定
      const nextDay = new Date(startDate);
      nextDay.setDate(nextDay.getDate() + 1);
      this.minEndDate = nextDay;
      console.log('minEndDate set to: ', this.minEndDate);
    } else {
      this.minEndDate = null;
      console.log('minEndDate reset to null');
    }
  }

}
